<?php
/* --------------------------------------------------------------
 SecureTokenRepository.php 2020-02-21
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Application\Token;

use DirectoryIterator;
use Exception;
use Gambio\Admin\Application\Token\Exceptions\SecureTokenNotFoundException;
use Gambio\Core\Application\ValueObjects\Path;
use IteratorIterator;
use SplFileInfo;
use Webmozart\Assert\Assert;
use function strpos;

/**
 * Class SecureTokenRepository
 * @package Gambio\Admin\Application\Token
 */
class SecureTokenRepository
{
    private const PREFIX = 'secure_token_';
    
    /**
     * @var string
     */
    private $mediaDir;
    
    /**
     * @var string
     */
    private $cachedToken;
    
    
    /**
     * SecureTokenRepository constructor.
     *
     * @param Path $path
     */
    public function __construct(Path $path)
    {
        $mediaDir = "{$path->base()}/media";
        Assert::directory($mediaDir);
        $this->mediaDir = $mediaDir;
    }
    
    
    /**
     * Provides the secure token.
     *
     * If no secure token is available, a new one will be created.
     * In case of subsequent method calls, the token will be internally cached.
     *
     * @return string
     */
    public function provideSecureToken(): string
    {
        if ($this->cachedToken) {
            return $this->cachedToken;
        }
        
        try {
            $this->cachedToken = $this->tryFetchToken();
            
            return $this->cachedToken;
        } catch (SecureTokenNotFoundException $e) {
            $this->cachedToken = $this->createNewToken();
            
            return $this->cachedToken;
        }
    }
    
    
    /**
     * Generates a new secure token.
     * This file additionally creates the secure token file.
     *
     * @return string
     */
    private function createNewToken(): string
    {
        try {
            $rng = random_int(0, PHP_INT_MAX);
        } catch (Exception $e) {
            $rng = mt_rand();
        }
        $token  = md5((string)$rng);
        $prefix = self::PREFIX;
        
        $tokenPath = "{$this->mediaDir}/{$prefix}{$token}";
        file_put_contents($tokenPath, '.');
        
        return $token;
    }
    
    
    /**
     * Tries to fetch the secure token from media/ directory.
     * An exception is thrown if the token was not found.
     *
     * @return string
     * @throws SecureTokenNotFoundException
     */
    private function tryFetchToken(): string
    {
        $iterator = new IteratorIterator(new DirectoryIterator($this->mediaDir));
        
        foreach ($iterator as $file) {
            /** @var SplFileInfo $file */
            if (strpos($file->getFilename(), self::PREFIX) !== false) {
                return str_replace(self::PREFIX, '', $file->getFilename());
            }
        }
        
        throw new SecureTokenNotFoundException();
    }
}